/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.android.media;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import android.app.Activity;
import android.media.MediaPlayer;
import android.media.MediaPlayer.OnCompletionListener;
import android.media.MediaPlayer.OnPreparedListener;
import android.os.Bundle;
import android.text.method.ScrollingMovementMethod;
import android.util.Log;
import android.webkit.URLUtil;
import android.widget.EditText;
import android.widget.MediaController;
import android.widget.TextView;
import android.widget.Toast;

import com.nielsen.vcbeacon.VCBeaconActivity;

public class VideoViewDemo extends Activity{
	
	//Just our basic container for marker positions
	class Marker {
		int startPos;
		int endPos;
		float tolerance = 0.015f;
		Marker(float location){
			//Since our tick count is only 1000ms we are setting a tolerance 
			//in case our current position doesn't fall exactly on our marker pos
			startPos = (int) (videoDuration * (location - tolerance)); 
			endPos = (int) (videoDuration * (location + tolerance));
		}
	}
	
	private static final String TAG = "VideoViewDemo";

	private CustomVideoView mVideoView;
	private EditText mPath;
	private TextView mMessageText;
	private String current;
	private VCBeaconActivity NVCBeacon;
	private MediaController mMediaCont;
	private boolean mPlaySent, mStopSent;
	private int highestPosition;
	int videoDuration = 0;
	private Marker[] myMarkers = {};
	
	

	@Override
	public void onCreate(Bundle icicle) {
		super.onCreate(icicle);
		setContentView(R.layout.main);
		
		mVideoView = (CustomVideoView) findViewById(R.id.custom_videoview);
		mVideoView.setVideoViewListener(new CustomVideoView.VideoViewListener() {

		    @Override
		    public void onPlay() {
		        System.out.println("Play!");
		        //Check to see if user has played through the video once already
		        //and this is a reset of the scrub bar (seekbar). If so, we reset our booleans
		        //and send our beacons again
		        if(mStopSent)
		        	mPlaySent = mStopSent = false;
		        
		        sendPlayBeacon();
		    }

		    @Override
		    public void onPause() {
		        System.out.println("Pause!");
		    }
		    
		    @Override
		    public void onSeekTo(int msec){
		    	System.out.println("Seek!");
		    }
		});
		
		//Setup our message textView so we can display the beacons being 
		//sent without using something like Fiddler
		mMessageText = (TextView) findViewById(R.id.MessageText);
		mMessageText.setMovementMethod(new ScrollingMovementMethod());
		
		//Clear out our message text on start
		mMessageText.clearComposingText();
		
		/*
		 * NOTE:
	 		When working with the VCBeacon, we want to use it like a queue.  We send our
	 		our creation for the queue by sending our "play" beacon (dav0).  We then only
	 		want to send the ping beacon (dav1) with the highest position received.  In the 
	 		code, we used markers to send the ping.  This ping beacon appends to the queue
	 		and allows us to only track the highest point in the video the user has watched.
	 		We finally send our stop beacon (dav2) upon the stop of the video leaving the
	 		page, which again symbolizes the user stopping their video watching.  The stop 
	 		beacon flushes the queue.
		*/
		//Instantiate our VCBeacon object to we can send our messages to the server
		NVCBeacon = new VCBeaconActivity(mMessageText);
		
		//Reset our play and stop booleans
		mPlaySent = mStopSent = false;
		
		//Reset our highest position
		highestPosition = 0;
		
		//Link up our mediaController to our videoView
		mMediaCont = new MediaController(this);
		mMediaCont.setPadding(0, 0, 0, 100);
		mVideoView.setMediaController(mMediaCont);
		mMediaCont.setAnchorView(mVideoView);
		
		//Setup a listener for the end of the video so we can send a stop beacon
		mVideoView.setOnCompletionListener(new OnCompletionListener() {
            @Override
            public void onCompletion(MediaPlayer mp) {
            	sendStopBeacon();
                Toast.makeText(getApplicationContext(), "Video completed", Toast.LENGTH_LONG).show();
            }
        });
		
		/* We use this to setup our markers because this method will be triggered
		when the video has been loaded. This allows for us to call getDuration 
		and have a valid value returned */
		mVideoView.setOnPreparedListener(new OnPreparedListener(){
			@Override
			public void onPrepared(MediaPlayer mp) {
				videoDuration = mVideoView.getDuration();
				 ArrayList<Marker> tempList = new ArrayList<Marker>();
				 tempList.add(new Marker(0.25f));
				 tempList.add(new Marker(0.50f));
				 tempList.add(new Marker(0.75f));
				 myMarkers = tempList.toArray(new Marker[tempList.size()]);
			}
		});
				
		mPath = (EditText) findViewById(R.id.path);
		mPath.setText("http://www.nielsenonlinesupport.com/clientsupport/source_video/test_movie_sound.3gp");
		
		//Set up our timer to track our video progress
		Timer progressTimer = new Timer();
		progressTimer.schedule(new TimerTask(){
			@Override
			public void run() {
				runOnUiThread(new Runnable(){
					public void run(){
						highestPosition = mVideoView.getCurrentPosition();
						if(mVideoView.isPlaying() && hitPositionMarker()){
							sendPingBeacon();
							
							}
					}
				});
			}
			
		}, 
		0, 1000);

		runOnUiThread(new Runnable(){
			public void run() {
				playVideo();
			}
			
		});
	}
	 
	 @Override
	 public void onStop(){
		 super.onStop();
		 //Trigger stop beacon when closed
		 sendStopBeacon();
	 }
	 
	private boolean hitPositionMarker(){
		if(myMarkers.length == 0)return false;
		
		int curPos = mVideoView.getCurrentPosition();
		return (curPos >= highestPosition &&
				(curPos > myMarkers[0].startPos && curPos < myMarkers[0].endPos) || //25%
				(curPos > myMarkers[1].startPos && curPos < myMarkers[1].endPos) || //50%
				(curPos > myMarkers[2].startPos && curPos < myMarkers[2].endPos) ); //75%
	}
	
	private void scrollToBottom(){
		try{
			if(mMessageText == null)return;
			int lineCnt = mMessageText.getLineCount();
			int lineTop = (lineCnt > 0) ? mMessageText.getLayout().getLineTop(lineCnt) : 0;
			
			final int scrollAmt = lineTop - mMessageText.getHeight();
			if(scrollAmt > 0)
				mMessageText.scrollTo(0, scrollAmt);
			else
				mMessageText.scrollTo(0, 0);
		}
		catch(Exception e){
			Log.e(TAG, "error: " + e.getMessage(), e);
		}
	}

	private void playVideo() {
		try {
			
			final String path = mPath.getText().toString();
			Log.v(TAG, "path: " + path);
			if (path == null || path.length() == 0) {
				Toast.makeText(VideoViewDemo.this, "File URL/path is empty",
						Toast.LENGTH_LONG).show();

			} else {
				// If the path has not changed, just start the media player
				if (path.equals(current) && mVideoView != null) {
					mVideoView.start();
					mVideoView.requestFocus();
					return;
				}
				current = path;
				mVideoView.setVideoPath(getDataSource(path));
				mVideoView.start();
				
				//Set a callback for when mVideoView is shown
				mVideoView.requestFocus();


			}
			
		} catch (Exception e) {
			Log.e(TAG, "error: " + e.getMessage(), e);
			if (mVideoView != null) {
				mVideoView.stopPlayback();
			}
		}
	}

	private String getDataSource(String path) throws IOException {
		if (!URLUtil.isNetworkUrl(path)) {
			return path;
		} else {
			URL url = new URL(path);
			URLConnection cn = url.openConnection();
			cn.connect();
			InputStream stream = cn.getInputStream();
			if (stream == null)
				throw new RuntimeException("stream is null");
			File temp = File.createTempFile("mediaplayertmp", "dat");
			temp.deleteOnExit();
			String tempPath = temp.getAbsolutePath();
			FileOutputStream out = new FileOutputStream(temp);
			byte buf[] = new byte[128];
			do {
				int numread = stream.read(buf);
				if (numread <= 0)
					break;
				out.write(buf, 0, numread);
			} while (true);
			try {
				stream.close();
			} catch (IOException ex) {
				Log.e(TAG, "error: " + ex.getMessage(), ex);
			}
			return tempPath;
		}
	}
	
	private void sendPlayBeacon(){
		try
		{
			//Already sent play beacon, so no need to send again
			if(mPlaySent)return;
			
			Map<String, String> clipData = new HashMap<String, String>();
			clipData.put("clientid", "us-502202");
			clipData.put("vcid","c24");
			clipData.put("sfcode", "us");
			NVCBeacon.initalize(clipData);
			clipData.put("title", "test + Title");
			clipData.put("cg", "test Cat");
			clipData.put("duration", "30");
			clipData.put("ou", "http://test.ou");
			clipData.put("cliptype", "ad");
			NVCBeacon.play(clipData); //this sends the config map to the beacon;
			mPlaySent = true;
			scrollToBottom();
		}
		catch(Exception e)
		{
			Log.e(TAG, "error: " + e.getMessage(), e);
		}

	}
	
	private void sendStopBeacon()
	{
		try
		{
			//Already sent stop beacon, so no need to send again
			if(mStopSent)return;
			
			NVCBeacon.stop(mVideoView.getCurrentPositionSec());
			mStopSent = true;
			scrollToBottom();
		}
		catch(Exception e)
		{
			Log.e(TAG, "error: " + e.getMessage(), e);
		}
	}
	
	private void sendPingBeacon()
	{
		try
		{
			NVCBeacon.ping(mVideoView.getCurrentPositionSec());	
			scrollToBottom();
		}
		catch(Exception e)
		{
			Log.e(TAG, "error: " + e.getMessage(), e);
		}
	}
	
}


